<?php
// config.php - Database configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'attendance');

require 'vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Reader\Csv;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;

// Database connection function
function getDBConnection() {
    try {
        $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    } catch(PDOException $e) {
        die("Connection failed: " . $e->getMessage());
    }
}

// Function to convert time to minutes
function convertTimeToMinutes($time_str) {
    if (empty($time_str)) return 0;
    
    // Handle "X min" format
    if (preg_match('/(\d+)\s*min/', $time_str, $matches)) {
        return (int)$matches[1];
    }
    
    // Handle "HH:MM:SS" format
    if (preg_match('/(\d+):(\d+):(\d+)/', $time_str, $matches)) {
        return (int)$matches[1] * 60 + (int)$matches[2];
    }
    
    // Handle "HH:MM" format
    if (preg_match('/(\d+):(\d+)/', $time_str, $matches)) {
        return (int)$matches[1] * 60 + (int)$matches[2];
    }
    
    return 0;
}

// Function to generate sample CSV file (No Composer needed!)
function generateSampleCSV() {
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment;filename="attendance_sample_template.csv"');
    header('Cache-Control: max-age=0');
    
    $output = fopen('php://output', 'w');
    
    // CSV Headers
    $headers = [
        'Person ID', 'Name', 'Department', 'Date', 'Timetable', 
        'Attendance Status', 'Check In', 'Check Out', 'Late', 
        'Early Leave', 'Attended', 'Absent', 'OT1', 'OT2'
    ];
    
    fputcsv($output, $headers);
    
    // Sample data
    $sampleData = [
        ['1001', 'John Doe', 'IT Department', '2024-01-15', '9:00-17:00', 'Present', '09:00:00', '17:00:00', '0 min', '0 min', '480 min', '0 min', '0 min', '0 min'],
        ['1002', 'Jane Smith', 'HR Department', '2024-01-15', '9:00-17:00', 'Late', '09:15:00', '17:00:00', '15 min', '0 min', '465 min', '0 min', '0 min', '0 min'],
        ['1003', 'Bob Johnson', 'Sales', '2024-01-15', '9:00-17:00', 'Present', '09:00:00', '16:30:00', '0 min', '30 min', '450 min', '0 min', '0 min', '0 min'],
        ['1004', 'Alice Brown', 'Marketing', '2024-01-15', '9:00-17:00', 'Absent', '-', '-', '0 min', '0 min', '0 min', '480 min', '0 min', '0 min'],
        ['1005', 'Charlie Wilson', 'IT Department', '2024-01-15', '9:00-17:00', 'Present', '09:00:00', '19:00:00', '0 min', '0 min', '480 min', '0 min', '120 min', '0 min'],
        ['1006', 'Diana Prince', 'Finance', '2024-01-16', '9:00-17:00', 'Present', '08:55:00', '17:00:00', '0 min', '0 min', '485 min', '0 min', '0 min', '0 min'],
        ['1007', 'Edward Norton', 'IT Department', '2024-01-16', '9:00-17:00', 'Late', '09:30:00', '17:00:00', '30 min', '0 min', '450 min', '0 min', '0 min', '0 min'],
        ['1008', 'Fiona Clark', 'HR Department', '2024-01-16', '9:00-17:00', 'Present', '09:00:00', '17:00:00', '0 min', '0 min', '480 min', '0 min', '0 min', '0 min'],
    ];
    
    foreach ($sampleData as $row) {
        fputcsv($output, $row);
    }
    
    fclose($output);
    exit;
}

// Handle sample download request
if (isset($_GET['download_sample'])) {
    generateSampleCSV();
}

// Function to import attendance data
function importAttendanceData() {
    // Check if PhpSpreadsheet is available
    $composerInstalled = file_exists('vendor/autoload.php');
    
    if (!$composerInstalled) {
        echo '<div class="message error">
                <strong>⚠️ PhpSpreadsheet Not Installed!</strong><br><br>
                To import Excel files, you need to install PhpSpreadsheet using Composer:<br><br>
                <code style="background:#f8f9fa; padding:10px; display:block; margin:10px 0; border-radius:5px;">
                    1. Open Command Prompt<br>
                    2. cd C:\xampp\htdocs\attendance<br>
                    3. composer require phpoffice/phpspreadsheet
                </code>
                <br>
                <strong>Alternative:</strong> Download the sample CSV file, edit it, and save as Excel (.xlsx) format using Microsoft Excel or Google Sheets.
              </div>';
        return;
    }
    
    require_once 'vendor/autoload.php';
    
    if (!isset($_FILES['excel_file']) || $_FILES['excel_file']['error'] !== UPLOAD_ERR_OK) {
        echo '<div class="message error">Error uploading file.</div>';
        return;
    }

    $file_path = $_FILES['excel_file']['tmp_name'];
    $file_type = pathinfo($_FILES['excel_file']['name'], PATHINFO_EXTENSION);

    try {
        // Create PhpSpreadsheet reader
        if (strtolower($file_type) === 'xlsx') {
            $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xlsx();
        } elseif (strtolower($file_type) === 'xls') {
            $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xls();
        } elseif (strtolower($file_type) === 'csv') {
            $reader = new \PhpOffice\PhpSpreadsheet\Reader\Csv();
        } else {
            echo '<div class="message error">Unsupported file format. Please use .xlsx, .xls, or .csv files.</div>';
            return;
        }
        
        $spreadsheet = $reader->load($file_path);
        $worksheet = $spreadsheet->getActiveSheet();
        
        $db = getDBConnection();
        $success_count = 0;
        $error_count = 0;
        $processed_data = [];

        // Start from row 2 (assuming row 1 is header)
        $highestRow = $worksheet->getHighestRow();
        
        for ($row = 2; $row <= $highestRow; $row++) {
            try {
                // Read cell values
                $person_id = trim($worksheet->getCell('A' . $row)->getCalculatedValue(), "' ");
                $name = $worksheet->getCell('B' . $row)->getCalculatedValue();
                $department = $worksheet->getCell('C' . $row)->getCalculatedValue();
                $date_str = $worksheet->getCell('D' . $row)->getCalculatedValue();
                $timetable = $worksheet->getCell('E' . $row)->getCalculatedValue();
                $attendance_status = $worksheet->getCell('F' . $row)->getCalculatedValue();
                $check_in = $worksheet->getCell('G' . $row)->getCalculatedValue();
                $check_out = $worksheet->getCell('H' . $row)->getCalculatedValue();
                $late = $worksheet->getCell('I' . $row)->getCalculatedValue();
                $early_leave = $worksheet->getCell('J' . $row)->getCalculatedValue();
                $attended = $worksheet->getCell('K' . $row)->getCalculatedValue();
                $absent = $worksheet->getCell('L' . $row)->getCalculatedValue();
                $ot1 = $worksheet->getCell('M' . $row)->getCalculatedValue();
                $ot2 = $worksheet->getCell('N' . $row)->getCalculatedValue();

                // Skip empty rows or header rows
                if (empty($person_id) || $person_id === 'Person ID') {
                    continue;
                }

                // Skip detail rows (Check-In Time rows)
                if (strpos($person_id, 'Check-In Time:') !== false) {
                    continue;
                }

                // Parse date - handle Excel date serial or string date
                if (is_numeric($date_str)) {
                    $attendance_date = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($date_str)->format('Y-m-d');
                } else {
                    $attendance_date = date('Y-m-d', strtotime($date_str));
                }

                // Clean check-in/check-out times
                $check_in = (!empty($check_in) && $check_in != '-') ? $check_in : null;
                $check_out = (!empty($check_out) && $check_out != '-') ? $check_out : null;

                // Convert time durations to minutes
                $late_minutes = convertTimeToMinutes($late);
                $early_leave_minutes = convertTimeToMinutes($early_leave);
                $attended_minutes = convertTimeToMinutes($attended);
                $absent_minutes = convertTimeToMinutes($absent);
                $ot1_minutes = convertTimeToMinutes($ot1);
                $ot2_minutes = convertTimeToMinutes($ot2);

                // Check if record already exists
                $stmt = $db->prepare("SELECT id FROM attendance_details WHERE person_id = ? AND attendance_date = ?");
                $stmt->execute([$person_id, $attendance_date]);
                
                if ($stmt->rowCount() > 0) {
                    // Update existing record
                    $sql = "UPDATE attendance_details SET 
                            name = ?, department = ?, timetable = ?, attendance_status = ?, 
                            check_in = ?, check_out = ?, late = ?, early_leave = ?, 
                            attended_minutes = ?, absent_minutes = ?, ot1_minutes = ?, ot2_minutes = ?
                            WHERE person_id = ? AND attendance_date = ?";
                    
                    $params = [$name, $department, $timetable, $attendance_status, 
                              $check_in, $check_out, $late_minutes, $early_leave_minutes, 
                              $attended_minutes, $absent_minutes, $ot1_minutes, $ot2_minutes, 
                              $person_id, $attendance_date];
                } else {
                    // Insert new record
                    $sql = "INSERT INTO attendance_details 
                            (person_id, name, department, attendance_date, timetable, 
                            attendance_status, check_in, check_out, late, early_leave, 
                            attended_minutes, absent_minutes, ot1_minutes, ot2_minutes) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    
                    $params = [$person_id, $name, $department, $attendance_date, $timetable, 
                              $attendance_status, $check_in, $check_out, $late_minutes, 
                              $early_leave_minutes, $attended_minutes, $absent_minutes, 
                              $ot1_minutes, $ot2_minutes];
                }

                $stmt = $db->prepare($sql);
                
                if ($stmt->execute($params)) {
                    $success_count++;
                    $processed_data[] = [
                        'person_id' => $person_id,
                        'name' => $name,
                        'date' => $attendance_date,
                        'status' => $attendance_status
                    ];
                } else {
                    $error_count++;
                }

            } catch (Exception $e) {
                $error_count++;
                error_log("Error processing row $row: " . $e->getMessage());
            }
        }

        // Display results
        echo "<div class='message success'>✅ Import completed!</div>";
        echo "<div class='message info'>📊 Successfully processed: $success_count records</div>";
        if ($error_count > 0) {
            echo "<div class='message error'>❌ Errors: $error_count records</div>";
        }

        // Show preview of imported data
        if (!empty($processed_data)) {
            echo "<h3>Imported Data Preview</h3>";
            echo "<table>";
            echo "<tr><th>Person ID</th><th>Name</th><th>Date</th><th>Status</th></tr>";
            foreach (array_slice($processed_data, 0, 10) as $data) {
                echo "<tr>";
                echo "<td>" . htmlspecialchars($data['person_id']) . "</td>";
                echo "<td>" . htmlspecialchars($data['name']) . "</td>";
                echo "<td>" . htmlspecialchars($data['date']) . "</td>";
                echo "<td>" . htmlspecialchars($data['status']) . "</td>";
                echo "</tr>";
            }
            echo "</table>";
            if (count($processed_data) > 10) {
                echo "<p>... and " . (count($processed_data) - 10) . " more records</p>";
            }
        }

    } catch (Exception $e) {
        echo "<div class='message error'>Error reading Excel file: " . $e->getMessage() . "</div>";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Attendance Data Import</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container { 
            max-width: 1000px; 
            margin: 0 auto;
            background: white;
            padding: 40px;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        
        h1 {
            color: #333;
            border-bottom: 4px solid #667eea;
            padding-bottom: 15px;
            margin-bottom: 30px;
            font-size: 32px;
        }
        
        .download-section {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 30px;
            border: 2px solid #667eea;
        }
        
        .download-section h3 {
            color: #333;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .download-section p {
            color: #555;
            margin-bottom: 15px;
            line-height: 1.6;
        }
        
        .btn-download {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        .btn-download:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
        }
        
        .composer-status {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 8px;
        }
        
        .composer-installed {
            background: #d4edda;
            border-left: 4px solid #28a745;
        }
        
        .form-group { 
            background: #f8f9fa;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 25px;
            border: 2px dashed #dee2e6;
        }
        
        label { 
            display: block; 
            margin-bottom: 10px; 
            font-weight: 600;
            color: #495057;
            font-size: 16px;
        }
        
        input[type="file"] { 
            padding: 12px; 
            border: 2px solid #dee2e6; 
            border-radius: 8px;
            width: 100%;
            background: white;
            cursor: pointer;
            font-size: 14px;
        }
        
        input[type="file"]:hover {
            border-color: #667eea;
        }
        
        button[type="submit"] { 
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
            color: white; 
            padding: 14px 35px; 
            border: none; 
            border-radius: 8px; 
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            margin-top: 15px;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        button[type="submit"]:hover { 
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(17, 153, 142, 0.4);
        }
        
        .message { 
            padding: 15px 20px; 
            margin: 20px 0; 
            border-radius: 8px;
            font-weight: 500;
            line-height: 1.6;
        }
        
        .success { 
            background: #d4edda; 
            color: #155724; 
            border-left: 4px solid #28a745;
        }
        
        .error { 
            background: #f8d7da; 
            color: #721c24; 
            border-left: 4px solid #dc3545;
        }
        
        .info { 
            background: #d1ecf1; 
            color: #0c5460; 
            border-left: 4px solid #17a2b8;
        }
        
        code {
            background: #f8f9fa;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            color: #e83e8c;
        }
        
        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin-top: 25px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            border-radius: 8px;
            overflow: hidden;
        }
        
        th, td { 
            border: 1px solid #dee2e6; 
            padding: 14px; 
            text-align: left; 
        }
        
        th { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 13px;
            letter-spacing: 0.5px;
        }
        
        tr:nth-child(even) {
            background-color: #f8f9fa;
        }
        
        tr:hover {
            background-color: #e9ecef;
        }
        
        .instructions {
            background: #e7f3ff;
            border-left: 4px solid #2196F3;
            border-radius: 8px;
            padding: 20px;
            margin: 25px 0;
        }
        
        .instructions h3 {
            margin-bottom: 15px;
            color: #1565C0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .instructions ul {
            margin: 15px 0;
            padding-left: 25px;
        }
        
        .instructions li {
            margin: 8px 0;
            line-height: 1.6;
            color: #1976D2;
        }
        
        .instructions strong {
            color: #0D47A1;
        }
        
        h3 {
            color: #333;
            margin: 25px 0 15px 0;
            font-size: 20px;
        }
        
        .icon {
            font-size: 20px;
        }
        
        .install-steps {
            background: white;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
        }
        
        .install-steps ol {
            padding-left: 20px;
        }
        
        .install-steps li {
            margin: 10px 0;
            line-height: 1.8;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📊 Attendance Data Import System</h1>
        
        <?php
        $composerInstalled = file_exists('vendor/autoload.php');
        if ($composerInstalled): ?>
            <div class="composer-status composer-installed">
                <strong>✅ System Ready!</strong> PhpSpreadsheet is installed and ready to import Excel files.
            </div>
        <?php else: ?>
            <div class="composer-status">
                <strong>⚠️ Setup Required:</strong> To import Excel files, you need to install PhpSpreadsheet using Composer.
                <div class="install-steps">
                    <ol>
                        <li>Download Composer from: <a href="https://getcomposer.org/download/" target="_blank">getcomposer.org</a></li>
                        <li>Open Command Prompt and navigate to: <code>cd C:\xampp\htdocs\attendance</code></li>
                        <li>Run: <code>composer require phpoffice/phpspreadsheet</code></li>
                        <li>Refresh this page</li>
                    </ol>
                </div>
                <strong>For now:</strong> You can download the CSV template below and convert it to Excel using Microsoft Excel or Google Sheets.
            </div>
        <?php endif; ?>
        
        <div class="download-section">
            <h3><span class="icon">📥</span> Download Sample Template</h3>
            <p>Download our pre-formatted CSV file with sample data. You can open it in Excel, edit the data, and save it as an Excel file (.xlsx) for import.</p>
            <a href="?download_sample=1" class="btn-download">
                <span class="icon">⬇️</span> Download Sample CSV Template
            </a>
        </div>
        
        <div class="instructions">
            <h3><span class="icon">📋</span> How to Use the Template</h3>
            <div class="install-steps">
                <ol>
                    <li><strong>Download</strong> the sample CSV template above</li>
                    <li><strong>Open</strong> the file in Microsoft Excel or Google Sheets</li>
                    <li><strong>Edit</strong> the sample data with your actual attendance records</li>
                    <li><strong>Save</strong> the file as Excel format (.xlsx or .xls)</li>
                    <li><strong>Upload</strong> the file using the form below</li>
                </ol>
            </div>
            <p><strong>Column Format:</strong></p>
            <ul>
                <li><strong>Column A:</strong> Person ID (e.g., 1001, 1002)</li>
                <li><strong>Column B:</strong> Name (e.g., John Doe)</li>
                <li><strong>Column C:</strong> Department (e.g., IT, HR, Sales)</li>
                <li><strong>Column D:</strong> Date (format: YYYY-MM-DD or MM/DD/YYYY)</li>
                <li><strong>Column E:</strong> Timetable (e.g., 9:00-17:00)</li>
                <li><strong>Column F:</strong> Attendance Status (e.g., Present, Absent, Late)</li>
                <li><strong>Column G-N:</strong> Time tracking data</li>
            </ul>
        </div>

        <div class="form-group">
            <form action="" method="post" enctype="multipart/form-data">
                <label for="excel_file"><span class="icon">📁</span> Select Excel or CSV File:</label>
                <input type="file" name="excel_file" id="excel_file" accept=".xlsx, .xls, .csv" required>
                <br>
                <button type="submit" name="import"><span class="icon">🚀</span> Import Data</button>
            </form>
        </div>

        <?php
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['import'])) {
            importAttendanceData();
        }
        ?>
    </div>
</body>
</html>